/*
 ---

 name: ServerConnector

 description: Provides an API for communication with the Applitools server.

 provides: [ServerConnector]
 requires: [GeneralUtils]

 ---
 */

(function () {
    "use strict";

    var GeneralUtils = require('eyes.utils').GeneralUtils,
        request = require('request'),
        fs = require('fs');

    // Constants
    var CONNECTION_TIMEOUT_MS = 5 * 60 * 1000,
        DEFAULT_HEADERS = {'Accept': 'application/json', 'Content-Type': 'application/json'},
        SERVER_SUFFIX = '/api/sessions/running';

    /**
     *
     * @param {PromiseFactory} promiseFactory An object which will be used for creating deferreds/promises.
     * @param {String} serverUri
     * @param {Object} logger
     * @constructor
     **/
    function ServerConnector(promiseFactory, serverUri, logger) {
        this._promiseFactory = promiseFactory;
        this._logger = logger;
        this._serverUri = GeneralUtils.urlConcat(serverUri, SERVER_SUFFIX);
        this._runKey = undefined;
        this._httpOptions = {
            strictSSL: false,
            headers: DEFAULT_HEADERS,
            timeout: CONNECTION_TIMEOUT_MS,
            qs: {}
        };
    }

    /**
     * Activate/Deactivate HTTP client debugging.
     *
     * @param {boolean} isDebug Whether or not to activate debugging.
     */
    ServerConnector.prototype.setDebugMode = function (isDebug) {
        request.debug = isDebug;
    };

    /**
     * @return {boolean} Whether or not debug mode is active.
     */
    ServerConnector.prototype.getIsDebugMode = function () {
        return request.debug;
    };

    /**
     * Sets the API key of your applitools Eyes account.
     *
     * @param runKey {String} The run key to be used.
     * @param newAuthScheme {boolean} Whether or not the server uses the new authentication scheme.
     */
    ServerConnector.prototype.setApiKey = function (runKey, newAuthScheme) {
        if (newAuthScheme) {
            this._httpOptions.qs.accessKey = runKey;
        } else {
            this._httpOptions.qs.apiKey = runKey;
        }
        this._runKey = runKey;
    };

    /**
     *
     * @return {String} The current run key.
     */
    ServerConnector.prototype.getApiKey = function () {
        return this._runKey;
    };

    /**
     * Whether sessions are removed immediately after they are finished.
     *
     * @param shouldRemove {boolean}
     */
    ServerConnector.prototype.setRemoveSession = function (shouldRemove) {
        this._httpOptions.qs.removeSession = shouldRemove;
    };

    /**
     *
     * @return {boolean} Whether sessions are removed immediately after they are finished.
     */
    ServerConnector.prototype.getRemoveSession = function () {
        return !!this._httpOptions.qs.removeSession;
    };

    /**
     *
     * Starts a new running session in the server. Based on the given parameters,
     * this running session will either be linked to an existing session, or to
     * a completely new session.
     *
     * @method startSession
     * @param {Object} sessionStartInfo - The start parameters for the session.
     * @return {Object} Promise with a resolve result that represents the current running session.
     *
     **/
    ServerConnector.prototype.startSession = function (sessionStartInfo) {
        this._logger.verbose('ServerConnector.startSession called with:', sessionStartInfo);
        return this._promiseFactory.makePromise(function (resolve, reject) {
            this._logger.verbose('ServerConnector.startSession will now post call');

            var options = Object.create(this._httpOptions);
            options.uri = this._serverUri;
            options.body = {startInfo: sessionStartInfo};
            options.json = true;
            options.method = "post";
            request(options, function (err, response, body) {
                if (err) {
                    this._logger.log('ServerConnector.startSession - post failed');
                    reject(new Error(err));
                    return;
                }

                this._logger.verbose('ServerConnector.startSession - start session result', body,
                    'status code ', response.statusCode);

                if (response.statusCode === 200 || response.statusCode === 201) {
                    this._logger.verbose('ServerConnector.startSession - post succeeded');
                    resolve({sessionId: body.id, legacySessionId: body.legacySessionId, sessionUrl: body.url,
                        isNewSession: response.statusCode === 201});
                }
            }.bind(this));
        }.bind(this));
    };

    /**
     *
     * Ends a running session in the server. Session results are received from the server.
     *
     * @method endSession
     * @param {Object} runningSession - The session to end.
     * @param {Object} isAborted.
     * @param {Object} save - Save the session.
     * @return {Object} Promise with a resolve result that represents the test results.
     *
     **/
    ServerConnector.prototype.endSession = function (runningSession, isAborted, save) {
        this._logger.verbose('ServerConnector.endSession called with isAborted:', isAborted,
            ', save:', save, 'for session:', runningSession);
        return this._promiseFactory.makePromise(function (resolve, reject) {
            var data = {aborted: isAborted, updateBaseline: save};
            var url = GeneralUtils.urlConcat(this._serverUri, runningSession.sessionId.toString());
            this._logger.verbose("ServerConnector.endSession will now post:", data, "to:", url);
            var options = Object.create(this._httpOptions);
            options.uri = GeneralUtils.urlConcat(this._serverUri, runningSession.sessionId.toString());
            options.qs.aborted = isAborted;
            options.qs.updateBaseline = save;
            options.json = true;
            options.method = 'delete';
            request(options, function (err, response, body) {
                if (err) {
                    this._logger.log('ServerConnector.endSession - delete failed');
                    reject(new Error(err));
                    return;
                }

                this._logger.verbose('ServerConnector.endSession result', body, 'status code', response.statusCode);
                if (response.statusCode === 200) {
                    resolve(body);
                } else {
                    reject(new Error(response));
                }
            }.bind(this));
        }.bind(this));
    };

    /**
     * Creates a bytes representation of the given JSON.
     * @param {object} jsonData The data from for which to create the bytes representation.
     * @return {Buffer} a buffer of bytes which represents the stringified JSON, prefixed with size.
     * @private
     */
    function _createDataBytes(jsonData) {
        var dataStr = JSON.stringify(jsonData);
        var dataLen = Buffer.byteLength(dataStr, 'utf8');
        // The result buffer will contain the length of the data + 4 bytes of size
        var result = new Buffer(dataLen + 4);
        result.writeUInt32BE(dataLen, 0);
        result.write(dataStr, 4, dataLen);
        return result;
    }

    ServerConnector.prototype.matchWindow = function (runningSession, matchWindowData, screenshot) {
        return this._promiseFactory.makePromise(function (resolve, reject) {
            var url = GeneralUtils.urlConcat(this._serverUri, runningSession.sessionId.toString());
            // TODO Daniel - Use binary image instead of base64 (see line below)
            //options.headers['Content-Type'] = 'application/octet-stream';
            //options.data = Buffer.concat([_createDataBytes(matchWindowData), screenshot]).toString('binary');
            this._logger.verbose("ServerConnector.matchWindow will now post to:", url);

            var options = Object.create(this._httpOptions);
            options.uri = url;
            options.body = matchWindowData;
            options.json = true;
            options.method = "post";
            request(options, function (err, response, body) {
                if (err) {
                    this._logger.log('ServerConnector.matchWindow - post failed');
                    reject(new Error(err));
                    return;
                }

                this._logger.verbose('ServerConnector.matchWindow result', body, 'status code', response.statusCode);
                if (response.statusCode === 200) {
                    resolve({asExpected: body.asExpected});
                } else {
                    reject(new Error(response));
                }
            }.bind(this));
        }.bind(this));
    };

    //noinspection JSValidateJSDoc
    /**
     * Replaces an actual image in the current running session.
     * @param {object} runningSession The currently running session.
     * @param {number} stepIndex The zero based index of the step in which to replace the actual image.
     * @param {object} replaceWindowData The updated window data (similar to matchWindowData only without ignoreMismatch).
     * @param {Buffer} screenshot The PNG bytes of the updated image.
     * @return {Promise} A promise which resolves when replacing is done, or rejects on error.
     */
    ServerConnector.prototype.replaceWindow = function (runningSession, stepIndex, replaceWindowData, screenshot) {
        return this._promiseFactory.makePromise(function (resolve, reject) {
            var url = GeneralUtils.urlConcat(this._serverUri, runningSession.sessionId.toString() + '/' + stepIndex);
            // TODO Daniel - Use binary image instead of base64 (see line below)
            //options.headers['Content-Type'] = 'application/octet-stream';
            //options.data = Buffer.concat([_createDataBytes(matchWindowData), screenshot]).toString('binary');
            this._logger.verbose("ServerConnector.replaceWindow will now post to:", url);
            var options = Object.create(this._httpOptions);
            options.uri = url;
            options.body = replaceWindowData;
            options.json = true;
            options.method = "put";
            request(options, function (err, response, body) {
                if (err) {
                    this._logger.log('ServerConnector.replaceWindow - put failed');
                    reject(new Error(err));
                    return;
                }

                this._logger.verbose('ServerConnector.replaceWindow result', body, 'status code', response.statusCode);
                if (response.statusCode === 200) {
                    resolve();
                } else {
                    reject(new Error(response));
                }
            }.bind(this));
        }.bind(this));
    };

    module.exports = ServerConnector;
}());
